/**
 * Utility function to get fallback image when Firebase image fails
 * @param {string} firebaseUrl - The Firebase Storage URL
 * @param {string} type - Type of image: 'banner', 'logo', 'blog', 'team'
 * @param {number} index - Index for selecting specific fallback image
 * @returns {string} - Fallback image path from public folder
 */
export const getImageFallback = (firebaseUrl, type = 'general', index = 0) => {
    // Check if URL is from Firebase Storage
    const isFirebaseUrl = firebaseUrl?.includes('firebasestorage.googleapis.com');
    
    if (!isFirebaseUrl) {
        // If not Firebase URL, return as is (might already be a local path)
        return firebaseUrl || '/images/general-placeholder.jpg';
    }

    // Fallback images based on type
    const fallbacks = {
        banner: [
            '/images/GCC20Comparison20Banner.webp', // Primary fallback for homepage hero section
        ],
        logo: [
            '/logos/appen.png',
            '/logos/denso.png',
            '/logos/isuzu.png',
            '/logos/microsoft.png',
            '/logos/nokia.png',
            '/logos/philips.png',
        ],
        blog: '/images/general-placeholder.jpg',
        team: '/images/team_member.png',
        general: '/images/general-placeholder.jpg',
    };

    if (type === 'banner' || type === 'logo') {
        const fallbackArray = fallbacks[type];
        const selectedIndex = index % fallbackArray.length;
        return fallbackArray[selectedIndex];
    }

    return fallbacks[type] || fallbacks.general;
};

/**
 * Check if image URL is from Firebase and should use fallback
 * @param {string} url - Image URL to check
 * @returns {boolean} - True if URL is from Firebase
 */
export const isFirebaseUrl = (url) => {
    return url?.includes('firebasestorage.googleapis.com') || false;
};

