'use client'
import { db } from '@/utils/firebase';
import { collection, getDocs } from 'firebase/firestore';
import { useLocale, useTranslations } from 'next-intl';
import Link from 'next/link';
import React, { useEffect, useState } from 'react'
import moment from 'moment';
import Image from 'next/image';
import { getImageFallback, isFirebaseUrl } from '@/utils/imageFallback';

const placeholderImage = '/images/general-placeholder.jpg';

const InsightCard = ({ limit = 0 }) => {
    const [blogs, setBlogs] = useState([]);
    const [imageErrors, setImageErrors] = useState({});
    const locale = useLocale();
    const t = useTranslations('Common');

    const fetchBlogs = async () => {
        const blogsCollection = collection(db, "Blog");
        const blogsSnapshot = await getDocs(blogsCollection);

        let blogs = blogsSnapshot.docs
            .map(doc => ({
                ...doc.data()
            }))
            .filter(blog => blog.status == true)

        blogs = blogs.sort((a, b) =>
            moment(b.date.en, "DD-MM-YYYY").isBefore(moment(a.date.en, "DD-MM-YYYY")) ? -1 : 1
        );

        if (limit) {
            blogs = blogs.slice(0, limit);
        }

        setBlogs(blogs);
    }

    useEffect(() => {
        fetchBlogs();
    }, [limit]);

    const handleImageError = (index, originalUrl) => {
        if (isFirebaseUrl(originalUrl)) {
            // Use fallback from public folder
            const fallback = getImageFallback(originalUrl, 'blog', index);
            setImageErrors(prev => {
                const newErrors = { ...prev };
                newErrors[index] = fallback;
                return newErrors;
            });
        } else {
            setImageErrors(prev => ({ ...prev, [index]: true }));
        }
    }

    return (
        <>
            {
                blogs?.map((item, index) => {
                    const originalImageSrc = locale == 'en' 
                        ? (item?.section_1_i || placeholderImage)
                        : (item?.section_2_i || placeholderImage);
                    
                    // Prioritize Firebase first, use fallback only if error occurred
                    let imageSrc = typeof imageErrors[index] === 'string' 
                        ? imageErrors[index]  // Use fallback if already set from error
                        : originalImageSrc;   // Otherwise use original (Firebase or placeholder)
                    
                    const hasError = imageErrors[index] === true;
                    
                    return (
                        <div key={index} className="col-span-12 md:col-span-6 lg:col-span-4 xl:col-span-3 mb-4 md:mb-0">
                            <Link href={locale == 'en' ? item?.section_3_i : item?.section_4_i} target='_blank'>
                                <div className='w-full overflow-hidden '>
                                    {hasError ? (
                                        <div className="w-full h-[300px] bg-gray-200 flex items-center justify-center">
                                            <span className="text-gray-500">Image unavailable</span>
                                        </div>
                                    ) : (
                                        <Image
                                            className='transform transition-transform duration-300 hover:scale-105'
                                            src={imageSrc}
                                            alt={locale == 'en' ? item?.title?.en : item?.title?.ar}
                                            width={500}
                                            height={300}
                                            style={{
                                                width: '100%',
                                                height: 'auto',
                                                objectFit: 'cover',
                                                objectPosition: 'center'
                                            }}
                                            onError={() => handleImageError(index, originalImageSrc)}
                                            loading="lazy"
                                            quality={80}
                                            placeholder="blur"
                                            blurDataURL="data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI0MDAiIGhlaWdodD0iMzAwIj48cmVjdCB3aWR0aD0iNDAwIiBoZWlnaHQ9IjMwMCIgZmlsbD0iI2VlZWVlZSIvPjwvc3ZnPg=="
                                            unoptimized={isFirebaseUrl(originalImageSrc)}
                                        />
                                    )}
                                </div>
                            <span className='inline-block mt-1 text-sm lg:text-base'>
                                {
                                    locale == 'en' ?
                                        item?.date?.en
                                        :
                                        item?.date?.ar
                                }
                            </span>
                            <h6 className='sub-heading mt-1'>
                                {
                                    locale == 'en' ?
                                        item?.title?.en
                                        :
                                        item?.title?.ar
                                }
                            </h6>
                            <span className='inline-block mt-2 text-sm lg:text-base'>
                                {t('written_by')} &nbsp;
                                {
                                    locale == 'en' ?
                                        item?.written_by?.en
                                        :
                                        item?.written_by?.ar
                                }
                            </span>
                            <div className='mt-3 flex items-center flex-wrap gap-2'>
                                {
                                    item?.keywords?.map((item, index) => (
                                        <div key={index} className='chip'>
                                            <span className='text-sm lg:text-base'>
                                                {
                                                    locale == 'en' ?
                                                        item?.en
                                                        :
                                                        item?.ar
                                                }
                                            </span>
                                        </div>
                                    ))
                                }
                            </div>
                        </Link>
                    </div>
                    );
                })
            }

        </>
    )
}

export default InsightCard